{"version":5,"vars":[{"line":154,"containerName":"","kind":2,"name":"base"},{"signature":{"parameters":[{"label":"$self"}],"documentation":"1;\n# $Id: Compatible.pm 16123 2009-09-17 12:57:27Z cjfields $\n#\n# BioPerl module for Bio::Tree::Compatible\n#\n# Please direct questions and support issues to <bioperl-l@bioperl.org> \n#\n# Cared for by Gabriel Valiente <valiente@lsi.upc.edu>\n#\n# Copyright Gabriel Valiente\n#\n# You may distribute this module under the same terms as Perl itself\n\n# POD documentation - main docs before the code\n\n=head1 NAME\n\nBio::Tree::Compatible - Testing compatibility of phylogenetic trees\nwith nested taxa.\n\n=head1 SYNOPSIS\n\n  use Bio::Tree::Compatible;\n  use Bio::TreeIO;\n  my $input = Bio::TreeIO->new('-format' => 'newick',\n                               '-file'   => 'input.tre');\n  my $t1 = $input->next_tree;\n  my $t2 = $input->next_tree;\n\n  my ($incompat, $ilabels, $inodes) = Bio::Tree::Compatible::is_compatible($t1,$t2);\n  if ($incompat) {\n    my %cluster1 = %{ Bio::Tree::Compatible::cluster_representation($t1) };\n    my %cluster2 = %{ Bio::Tree::Compatible::cluster_representation($t2) };\n    print \"incompatible trees\\n\";\n    if (scalar(@$ilabels)) {\n      foreach my $label (@$ilabels) {\n        my $node1 = $t1->find_node(-id => $label);\n        my $node2 = $t2->find_node(-id => $label);\n        my @c1 = sort @{ $cluster1{$node1} };\n        my @c2 = sort @{ $cluster2{$node2} };\n        print \"label $label\";\n        print \" cluster\"; map { print \" \",$_ } @c1;\n        print \" cluster\"; map { print \" \",$_ } @c2; print \"\\n\";\n      }\n    }\n    if (scalar(@$inodes)) {\n      while (@$inodes) {\n        my $node1 = shift @$inodes;\n        my $node2 = shift @$inodes;\n        my @c1 = sort @{ $cluster1{$node1} };\n        my @c2 = sort @{ $cluster2{$node2} };\n        print \"cluster\"; map { print \" \",$_ } @c1;\n        print \" properly intersects cluster\";\n        map { print \" \",$_ } @c2; print \"\\n\";\n      }\n    }\n  } else {\n    print \"compatible trees\\n\";\n  }\n\n=head1 DESCRIPTION\n\nNB: This module has exclusively class methods that work on Bio::Tree::TreeI\nobjects. An instance of Bio::Tree::Compatible cannot itself represent a tree,\nand so typically there is no need to create one.\n\nBio::Tree::Compatible is a Perl tool for testing compatibility of\nphylogenetic trees with nested taxa represented as Bio::Tree::Tree\nobjects. It is based on a recent characterization of ancestral\ncompatibility of semi-labeled trees in terms of their cluster\nrepresentations.\n\nA semi-labeled tree is a phylogenetic tree with some of its internal\nnodes labeled, and it can represent a classification tree as well as a\nphylogenetic tree with nested taxa, with labeled internal nodes\ncorresponding to taxa at a higher level of aggregation or nesting than\nthat of their descendents.\n\nTwo semi-labeled trees are compatible if their topological\nrestrictions to the common labels are such that for each node label,\nthe smallest clusters containing it in each of the trees coincide and,\nfurthermore, no cluster in one of the trees properly intersects a\ncluster of the other tree.\n\nFuture extensions of Bio::Tree::Compatible include a\nBio::Tree::Supertree module for combining compatible phylogenetic\ntrees with nested taxa into a common supertree.\n\n=head1 FEEDBACK\n\n=head2 Mailing Lists\n\nUser feedback is an integral part of the evolution of this and other\nBioperl modules. Send your comments and suggestions preferably to the\nBioperl mailing list.  Your participation is much appreciated.\n\n  bioperl-l@bioperl.org                  - General discussion\n  http://bioperl.org/wiki/Mailing_lists  - About the mailing lists\n\n=head2 Support \n\nPlease direct usage questions or support issues to the mailing list:\n\nI<bioperl-l@bioperl.org>\n\nrather than to the module maintainer directly. Many experienced and \nreponsive experts will be able look at the problem and quickly \naddress it. Please include a thorough description of the problem \nwith code and data examples if at all possible.\n\n=head2 Reporting Bugs\n\nReport bugs to the Bioperl bug tracking system to help us keep track\nof the bugs and their resolution. Bug reports can be submitted via the\nweb:\n\n  http://bugzilla.open-bio.org/\n\n=head1 SEE ALSO\n\n\n* * Philip Daniel and Charles Semple. Supertree Algorithms for\nNested Taxa. In: Olaf R. P. Bininda-Emonds (ed.) Phylogenetic\nSupertrees: Combining Information to Reveal the Tree of Life,\nI<Computational Biology>, vol. 4, chap. 7, pp. 151-171. Kluwer (2004).\n\n* * Charles Semple, Philip Daniel, Wim Hordijk, Roderic\nD. M. Page, and Mike Steel: Supertree Algorithms for Ancestral\nDivergence Dates and Nested Taxa. Bioinformatics B<20>(15), 2355-2360\n(2004).\n\n* * Merce Llabres, Jairo Rocha, Francesc Rossello, and Gabriel\nValiente: On the Ancestral Compatibility of Two Phylogenetic Trees\nwith Nested Taxa. J. Math. Biol. B<53>(3), 340-364 (2006).\n\n\n=head1 AUTHOR - Gabriel Valiente\n\nEmail valiente@lsi.upc.edu\n\n=head1 APPENDIX\n\nThe rest of the documentation details each of the object methods.\n\n\npackage Bio::Tree::Compatible;\nuse strict;\n\n# Object preamble - inherits from Bio::Root::Root\n\nuse Set::Scalar;\n\nuse base qw(Bio::Root::Root);\n\n=head2 postorder_traversal\n\n Title   : postorder_traversal\n Usage   : my @nodes = @{ $tree->postorder_traversal }\n Function: Return list of nodes in postorder\n Returns : reference to array of Bio::Tree::Node\n Args    : none\n\nFor example, the postorder traversal of the tree\nC<(((A,B)C,D),(E,F,G));> is a reference to an array of nodes with\ninternal_id 0 through 9, because the Newick standard representation\nfor phylogenetic trees is based on a postorder traversal.\n\n          +---A                    +---0\n          |                        |\n  +---+---C                +---4---2\n  |   |   |                |   |   |\n  |   |   +---B            |   |   +---1\n  |   |                    |   |\n  +   +-------D            9   +-------3\n  |                        |\n  |     +-----E            |     +-----5\n  |     |                  |     |\n  +-----+-----F            +-----8-----6\n        |                        |\n        +-----G                  +-----7","label":"postorder_traversal($self)"},"line":185,"range":{"start":{"character":0,"line":185},"end":{"line":199,"character":9999}},"kind":12,"definition":"sub","detail":"($self)","children":[{"definition":"my","line":186,"localvar":"my","containerName":"postorder_traversal","kind":13,"name":"$self"},{"line":187,"containerName":"postorder_traversal","localvar":"my","kind":13,"name":"@stack","definition":"my"},{"definition":"my","name":"@queue","localvar":"my","containerName":"postorder_traversal","kind":13,"line":188},{"line":189,"kind":13,"containerName":"postorder_traversal","name":"@stack"},{"line":189,"kind":13,"containerName":"postorder_traversal","name":"$self"},{"name":"get_root_node","kind":12,"containerName":"postorder_traversal","line":189},{"kind":13,"containerName":"postorder_traversal","name":"@stack","line":190},{"definition":"my","name":"$node","localvar":"my","kind":13,"containerName":"postorder_traversal","line":191},{"containerName":"postorder_traversal","kind":13,"name":"@stack","line":191},{"line":192,"containerName":"postorder_traversal","kind":13,"name":"@queue"},{"line":192,"name":"$node","kind":13,"containerName":"postorder_traversal"},{"definition":"my","line":193,"kind":13,"localvar":"my","containerName":"postorder_traversal","name":"$child"},{"line":193,"name":"$node","containerName":"postorder_traversal","kind":13},{"kind":12,"containerName":"postorder_traversal","name":"each_Descendent","line":193},{"line":194,"name":"@stack","kind":13,"containerName":"postorder_traversal"},{"line":194,"containerName":"postorder_traversal","kind":13,"name":"$child"},{"definition":"my","line":197,"name":"@postorder","localvar":"my","containerName":"postorder_traversal","kind":13},{"line":197,"name":"@queue","kind":13,"containerName":"postorder_traversal"},{"containerName":"postorder_traversal","kind":13,"name":"@postorder","line":198}],"containerName":"main::","name":"postorder_traversal"},{"range":{"end":{"character":9999,"line":230},"start":{"character":0,"line":227}},"kind":12,"line":227,"signature":{"parameters":[{"label":"$tree"}],"documentation":"1;\n# $Id: Compatible.pm 16123 2009-09-17 12:57:27Z cjfields $\n#\n# BioPerl module for Bio::Tree::Compatible\n#\n# Please direct questions and support issues to <bioperl-l@bioperl.org> \n#\n# Cared for by Gabriel Valiente <valiente@lsi.upc.edu>\n#\n# Copyright Gabriel Valiente\n#\n# You may distribute this module under the same terms as Perl itself\n\n# POD documentation - main docs before the code\n\n=head1 NAME\n\nBio::Tree::Compatible - Testing compatibility of phylogenetic trees\nwith nested taxa.\n\n=head1 SYNOPSIS\n\n  use Bio::Tree::Compatible;\n  use Bio::TreeIO;\n  my $input = Bio::TreeIO->new('-format' => 'newick',\n                               '-file'   => 'input.tre');\n  my $t1 = $input->next_tree;\n  my $t2 = $input->next_tree;\n\n  my ($incompat, $ilabels, $inodes) = Bio::Tree::Compatible::is_compatible($t1,$t2);\n  if ($incompat) {\n    my %cluster1 = %{ Bio::Tree::Compatible::cluster_representation($t1) };\n    my %cluster2 = %{ Bio::Tree::Compatible::cluster_representation($t2) };\n    print \"incompatible trees\\n\";\n    if (scalar(@$ilabels)) {\n      foreach my $label (@$ilabels) {\n        my $node1 = $t1->find_node(-id => $label);\n        my $node2 = $t2->find_node(-id => $label);\n        my @c1 = sort @{ $cluster1{$node1} };\n        my @c2 = sort @{ $cluster2{$node2} };\n        print \"label $label\";\n        print \" cluster\"; map { print \" \",$_ } @c1;\n        print \" cluster\"; map { print \" \",$_ } @c2; print \"\\n\";\n      }\n    }\n    if (scalar(@$inodes)) {\n      while (@$inodes) {\n        my $node1 = shift @$inodes;\n        my $node2 = shift @$inodes;\n        my @c1 = sort @{ $cluster1{$node1} };\n        my @c2 = sort @{ $cluster2{$node2} };\n        print \"cluster\"; map { print \" \",$_ } @c1;\n        print \" properly intersects cluster\";\n        map { print \" \",$_ } @c2; print \"\\n\";\n      }\n    }\n  } else {\n    print \"compatible trees\\n\";\n  }\n\n=head1 DESCRIPTION\n\nNB: This module has exclusively class methods that work on Bio::Tree::TreeI\nobjects. An instance of Bio::Tree::Compatible cannot itself represent a tree,\nand so typically there is no need to create one.\n\nBio::Tree::Compatible is a Perl tool for testing compatibility of\nphylogenetic trees with nested taxa represented as Bio::Tree::Tree\nobjects. It is based on a recent characterization of ancestral\ncompatibility of semi-labeled trees in terms of their cluster\nrepresentations.\n\nA semi-labeled tree is a phylogenetic tree with some of its internal\nnodes labeled, and it can represent a classification tree as well as a\nphylogenetic tree with nested taxa, with labeled internal nodes\ncorresponding to taxa at a higher level of aggregation or nesting than\nthat of their descendents.\n\nTwo semi-labeled trees are compatible if their topological\nrestrictions to the common labels are such that for each node label,\nthe smallest clusters containing it in each of the trees coincide and,\nfurthermore, no cluster in one of the trees properly intersects a\ncluster of the other tree.\n\nFuture extensions of Bio::Tree::Compatible include a\nBio::Tree::Supertree module for combining compatible phylogenetic\ntrees with nested taxa into a common supertree.\n\n=head1 FEEDBACK\n\n=head2 Mailing Lists\n\nUser feedback is an integral part of the evolution of this and other\nBioperl modules. Send your comments and suggestions preferably to the\nBioperl mailing list.  Your participation is much appreciated.\n\n  bioperl-l@bioperl.org                  - General discussion\n  http://bioperl.org/wiki/Mailing_lists  - About the mailing lists\n\n=head2 Support \n\nPlease direct usage questions or support issues to the mailing list:\n\nI<bioperl-l@bioperl.org>\n\nrather than to the module maintainer directly. Many experienced and \nreponsive experts will be able look at the problem and quickly \naddress it. Please include a thorough description of the problem \nwith code and data examples if at all possible.\n\n=head2 Reporting Bugs\n\nReport bugs to the Bioperl bug tracking system to help us keep track\nof the bugs and their resolution. Bug reports can be submitted via the\nweb:\n\n  http://bugzilla.open-bio.org/\n\n=head1 SEE ALSO\n\n\n* * Philip Daniel and Charles Semple. Supertree Algorithms for\nNested Taxa. In: Olaf R. P. Bininda-Emonds (ed.) Phylogenetic\nSupertrees: Combining Information to Reveal the Tree of Life,\nI<Computational Biology>, vol. 4, chap. 7, pp. 151-171. Kluwer (2004).\n\n* * Charles Semple, Philip Daniel, Wim Hordijk, Roderic\nD. M. Page, and Mike Steel: Supertree Algorithms for Ancestral\nDivergence Dates and Nested Taxa. Bioinformatics B<20>(15), 2355-2360\n(2004).\n\n* * Merce Llabres, Jairo Rocha, Francesc Rossello, and Gabriel\nValiente: On the Ancestral Compatibility of Two Phylogenetic Trees\nwith Nested Taxa. J. Math. Biol. B<53>(3), 340-364 (2006).\n\n\n=head1 AUTHOR - Gabriel Valiente\n\nEmail valiente@lsi.upc.edu\n\n=head1 APPENDIX\n\nThe rest of the documentation details each of the object methods.\n\n\npackage Bio::Tree::Compatible;\nuse strict;\n\n# Object preamble - inherits from Bio::Root::Root\n\nuse Set::Scalar;\n\nuse base qw(Bio::Root::Root);\n\n=head2 postorder_traversal\n\n Title   : postorder_traversal\n Usage   : my @nodes = @{ $tree->postorder_traversal }\n Function: Return list of nodes in postorder\n Returns : reference to array of Bio::Tree::Node\n Args    : none\n\nFor example, the postorder traversal of the tree\nC<(((A,B)C,D),(E,F,G));> is a reference to an array of nodes with\ninternal_id 0 through 9, because the Newick standard representation\nfor phylogenetic trees is based on a postorder traversal.\n\n          +---A                    +---0\n          |                        |\n  +---+---C                +---4---2\n  |   |   |                |   |   |\n  |   |   +---B            |   |   +---1\n  |   |                    |   |\n  +   +-------D            9   +-------3\n  |                        |\n  |     +-----E            |     +-----5\n  |     |                  |     |\n  +-----+-----F            +-----8-----6\n        |                        |\n        +-----G                  +-----7\n\n\nsub postorder_traversal {\n  my($self) = @_;\n  my @stack;\n  my @queue;\n  push @stack, $self->get_root_node;\n  while (@stack) {\n    my $node = pop @stack;\n    push @queue, $node;\n    foreach my $child ($node->each_Descendent(-sortby => 'internal_id')) {\n      push @stack, $child;\n    }\n  }\n  my @postorder = reverse @queue;\n  return \\@postorder;\n}\n\n=head2 cluster_representation\n\n Title   : cluster_representation\n Usage   : my %cluster = %{ $tree->cluster_representation }\n Function: Compute the cluster representation of a tree\n Returns : reference to hash of array of string indexed by\n           Bio::Tree::Node\n Args    : none\n\nFor example, the cluster representation of the tree\nC<(((A,B)C,D),(E,F,G));> is a reference to a hash associating an array\nof string (descendent labels) to each node, as follows:\n\n  0 --> [A]\n  1 --> [B]\n  2 --> [A,B,C]\n  3 --> [D]\n  4 --> [A,B,C,D]\n  5 --> [E]\n  6 --> [F]\n  7 --> [G]\n  8 --> [E,F,G]\n  9 --> [A,B,C,D,E,F,G]","label":"cluster_representation($tree)"},"containerName":"main::","name":"cluster_representation","children":[{"definition":"my","line":228,"localvar":"my","containerName":"cluster_representation","kind":13,"name":"$tree"},{"name":"%cluster","localvar":"my","kind":13,"containerName":"cluster_representation","line":229,"definition":"my"},{"kind":13,"localvar":"my","containerName":"cluster_representation","name":"@postorder","line":230,"definition":"my"},{"line":230,"name":"$tree","containerName":"cluster_representation","kind":13}],"detail":"($tree)","definition":"sub"},{"definition":"my","localvar":"my","kind":13,"containerName":null,"name":"$node","line":231},{"name":"@postorder","containerName":null,"kind":13,"line":231},{"containerName":null,"localvar":"my","kind":13,"name":"@labeled","line":232,"definition":"my"},{"name":"id","kind":12,"containerName":"main::","line":232},{"name":"id","containerName":"main::","kind":12,"line":232},{"kind":13,"containerName":null,"name":"$node","line":232},{"containerName":"main::","kind":12,"name":"get_Descendents","line":232},{"name":"@labeled","containerName":null,"kind":13,"line":233},{"name":"$node","kind":13,"containerName":null,"line":233},{"line":233,"containerName":"main::","kind":12,"name":"id"},{"name":"$node","containerName":null,"kind":13,"line":233},{"name":"id","containerName":"main::","kind":12,"line":233},{"containerName":null,"kind":13,"name":"%cluster","line":234},{"line":234,"name":"$node","containerName":null,"kind":13},{"name":"@labeled","containerName":null,"kind":13,"line":234},{"kind":13,"containerName":null,"name":"%cluster","line":236},{"signature":{"parameters":[{"label":"$self"},{"label":"$arg"}],"documentation":"1;\n# $Id: Compatible.pm 16123 2009-09-17 12:57:27Z cjfields $\n#\n# BioPerl module for Bio::Tree::Compatible\n#\n# Please direct questions and support issues to <bioperl-l@bioperl.org> \n#\n# Cared for by Gabriel Valiente <valiente@lsi.upc.edu>\n#\n# Copyright Gabriel Valiente\n#\n# You may distribute this module under the same terms as Perl itself\n\n# POD documentation - main docs before the code\n\n=head1 NAME\n\nBio::Tree::Compatible - Testing compatibility of phylogenetic trees\nwith nested taxa.\n\n=head1 SYNOPSIS\n\n  use Bio::Tree::Compatible;\n  use Bio::TreeIO;\n  my $input = Bio::TreeIO->new('-format' => 'newick',\n                               '-file'   => 'input.tre');\n  my $t1 = $input->next_tree;\n  my $t2 = $input->next_tree;\n\n  my ($incompat, $ilabels, $inodes) = Bio::Tree::Compatible::is_compatible($t1,$t2);\n  if ($incompat) {\n    my %cluster1 = %{ Bio::Tree::Compatible::cluster_representation($t1) };\n    my %cluster2 = %{ Bio::Tree::Compatible::cluster_representation($t2) };\n    print \"incompatible trees\\n\";\n    if (scalar(@$ilabels)) {\n      foreach my $label (@$ilabels) {\n        my $node1 = $t1->find_node(-id => $label);\n        my $node2 = $t2->find_node(-id => $label);\n        my @c1 = sort @{ $cluster1{$node1} };\n        my @c2 = sort @{ $cluster2{$node2} };\n        print \"label $label\";\n        print \" cluster\"; map { print \" \",$_ } @c1;\n        print \" cluster\"; map { print \" \",$_ } @c2; print \"\\n\";\n      }\n    }\n    if (scalar(@$inodes)) {\n      while (@$inodes) {\n        my $node1 = shift @$inodes;\n        my $node2 = shift @$inodes;\n        my @c1 = sort @{ $cluster1{$node1} };\n        my @c2 = sort @{ $cluster2{$node2} };\n        print \"cluster\"; map { print \" \",$_ } @c1;\n        print \" properly intersects cluster\";\n        map { print \" \",$_ } @c2; print \"\\n\";\n      }\n    }\n  } else {\n    print \"compatible trees\\n\";\n  }\n\n=head1 DESCRIPTION\n\nNB: This module has exclusively class methods that work on Bio::Tree::TreeI\nobjects. An instance of Bio::Tree::Compatible cannot itself represent a tree,\nand so typically there is no need to create one.\n\nBio::Tree::Compatible is a Perl tool for testing compatibility of\nphylogenetic trees with nested taxa represented as Bio::Tree::Tree\nobjects. It is based on a recent characterization of ancestral\ncompatibility of semi-labeled trees in terms of their cluster\nrepresentations.\n\nA semi-labeled tree is a phylogenetic tree with some of its internal\nnodes labeled, and it can represent a classification tree as well as a\nphylogenetic tree with nested taxa, with labeled internal nodes\ncorresponding to taxa at a higher level of aggregation or nesting than\nthat of their descendents.\n\nTwo semi-labeled trees are compatible if their topological\nrestrictions to the common labels are such that for each node label,\nthe smallest clusters containing it in each of the trees coincide and,\nfurthermore, no cluster in one of the trees properly intersects a\ncluster of the other tree.\n\nFuture extensions of Bio::Tree::Compatible include a\nBio::Tree::Supertree module for combining compatible phylogenetic\ntrees with nested taxa into a common supertree.\n\n=head1 FEEDBACK\n\n=head2 Mailing Lists\n\nUser feedback is an integral part of the evolution of this and other\nBioperl modules. Send your comments and suggestions preferably to the\nBioperl mailing list.  Your participation is much appreciated.\n\n  bioperl-l@bioperl.org                  - General discussion\n  http://bioperl.org/wiki/Mailing_lists  - About the mailing lists\n\n=head2 Support \n\nPlease direct usage questions or support issues to the mailing list:\n\nI<bioperl-l@bioperl.org>\n\nrather than to the module maintainer directly. Many experienced and \nreponsive experts will be able look at the problem and quickly \naddress it. Please include a thorough description of the problem \nwith code and data examples if at all possible.\n\n=head2 Reporting Bugs\n\nReport bugs to the Bioperl bug tracking system to help us keep track\nof the bugs and their resolution. Bug reports can be submitted via the\nweb:\n\n  http://bugzilla.open-bio.org/\n\n=head1 SEE ALSO\n\n\n* * Philip Daniel and Charles Semple. Supertree Algorithms for\nNested Taxa. In: Olaf R. P. Bininda-Emonds (ed.) Phylogenetic\nSupertrees: Combining Information to Reveal the Tree of Life,\nI<Computational Biology>, vol. 4, chap. 7, pp. 151-171. Kluwer (2004).\n\n* * Charles Semple, Philip Daniel, Wim Hordijk, Roderic\nD. M. Page, and Mike Steel: Supertree Algorithms for Ancestral\nDivergence Dates and Nested Taxa. Bioinformatics B<20>(15), 2355-2360\n(2004).\n\n* * Merce Llabres, Jairo Rocha, Francesc Rossello, and Gabriel\nValiente: On the Ancestral Compatibility of Two Phylogenetic Trees\nwith Nested Taxa. J. Math. Biol. B<53>(3), 340-364 (2006).\n\n\n=head1 AUTHOR - Gabriel Valiente\n\nEmail valiente@lsi.upc.edu\n\n=head1 APPENDIX\n\nThe rest of the documentation details each of the object methods.\n\n\npackage Bio::Tree::Compatible;\nuse strict;\n\n# Object preamble - inherits from Bio::Root::Root\n\nuse Set::Scalar;\n\nuse base qw(Bio::Root::Root);\n\n=head2 postorder_traversal\n\n Title   : postorder_traversal\n Usage   : my @nodes = @{ $tree->postorder_traversal }\n Function: Return list of nodes in postorder\n Returns : reference to array of Bio::Tree::Node\n Args    : none\n\nFor example, the postorder traversal of the tree\nC<(((A,B)C,D),(E,F,G));> is a reference to an array of nodes with\ninternal_id 0 through 9, because the Newick standard representation\nfor phylogenetic trees is based on a postorder traversal.\n\n          +---A                    +---0\n          |                        |\n  +---+---C                +---4---2\n  |   |   |                |   |   |\n  |   |   +---B            |   |   +---1\n  |   |                    |   |\n  +   +-------D            9   +-------3\n  |                        |\n  |     +-----E            |     +-----5\n  |     |                  |     |\n  +-----+-----F            +-----8-----6\n        |                        |\n        +-----G                  +-----7\n\n\nsub postorder_traversal {\n  my($self) = @_;\n  my @stack;\n  my @queue;\n  push @stack, $self->get_root_node;\n  while (@stack) {\n    my $node = pop @stack;\n    push @queue, $node;\n    foreach my $child ($node->each_Descendent(-sortby => 'internal_id')) {\n      push @stack, $child;\n    }\n  }\n  my @postorder = reverse @queue;\n  return \\@postorder;\n}\n\n=head2 cluster_representation\n\n Title   : cluster_representation\n Usage   : my %cluster = %{ $tree->cluster_representation }\n Function: Compute the cluster representation of a tree\n Returns : reference to hash of array of string indexed by\n           Bio::Tree::Node\n Args    : none\n\nFor example, the cluster representation of the tree\nC<(((A,B)C,D),(E,F,G));> is a reference to a hash associating an array\nof string (descendent labels) to each node, as follows:\n\n  0 --> [A]\n  1 --> [B]\n  2 --> [A,B,C]\n  3 --> [D]\n  4 --> [A,B,C,D]\n  5 --> [E]\n  6 --> [F]\n  7 --> [G]\n  8 --> [E,F,G]\n  9 --> [A,B,C,D,E,F,G]\n\n\nsub cluster_representation {\n  my ($tree) = @_;\n  my %cluster;\n  my @postorder = @{ postorder_traversal($tree) };\n  foreach my $node ( @postorder ) {\n    my @labeled = map { $_->id } grep { $_->id } $node->get_Descendents;\n    push @labeled, $node->id if $node->id;\n    $cluster{$node} = \\@labeled;\n  }\n  return \\%cluster;\n}\n\n=head2 common_labels\n\n Title   : common_labels\n Usage   : my $labels = $tree1->common_labels($tree2);\n Function: Return set of common node labels\n Returns : Set::Scalar\n Args    : Bio::Tree::Tree\n\nFor example, the common labels of the tree C<(((A,B)C,D),(E,F,G));>\nand the tree C<((A,B)H,E,(J,(K)G)I);> are: C<[A,B,E,G]>.\n\n          +---A                 +---A\n          |                     |\n  +---+---C             +-------H\n  |   |   |             |       |\n  |   |   +---B         |       +---B\n  |   |                 |\n  +   +-------D         +-----------E\n  |                     |\n  |     +-----E         |   +-------J\n  |     |               |   |\n  +-----+-----F         +---I\n        |                   |\n        +-----G             +---G---K","label":"common_labels($self,$arg)"},"line":266,"kind":12,"range":{"start":{"character":0,"line":266},"end":{"character":9999,"line":273}},"definition":"sub","detail":"($self,$arg)","children":[{"localvar":"my","kind":13,"containerName":"common_labels","name":"$self","line":267,"definition":"my"},{"line":267,"containerName":"common_labels","kind":13,"name":"$arg"},{"definition":"my","line":268,"kind":13,"localvar":"my","containerName":"common_labels","name":"@labels1"},{"containerName":"common_labels","kind":12,"name":"id","line":268},{"name":"id","kind":12,"containerName":"common_labels","line":268},{"line":268,"name":"$self","containerName":"common_labels","kind":13},{"kind":12,"containerName":"common_labels","name":"get_nodes","line":268},{"definition":"my","containerName":"common_labels","localvar":"my","kind":13,"name":"$common","line":269},{"line":269,"kind":12,"containerName":"common_labels","name":"new"},{"name":"@labels1","containerName":"common_labels","kind":13,"line":269},{"localvar":"my","containerName":"common_labels","kind":13,"name":"@labels2","line":270,"definition":"my"},{"line":270,"name":"id","containerName":"common_labels","kind":12},{"name":"id","containerName":"common_labels","kind":12,"line":270},{"name":"$arg","containerName":"common_labels","kind":13,"line":270},{"line":270,"name":"get_nodes","kind":12,"containerName":"common_labels"},{"definition":"my","line":271,"name":"$temp","containerName":"common_labels","localvar":"my","kind":13},{"line":271,"name":"new","containerName":"common_labels","kind":12},{"line":271,"kind":13,"containerName":"common_labels","name":"@labels2"},{"line":272,"name":"$common","kind":13,"containerName":"common_labels"},{"line":272,"name":"intersection","kind":12,"containerName":"common_labels"},{"line":272,"kind":13,"containerName":"common_labels","name":"$temp"}],"containerName":"main::","name":"common_labels"},{"kind":12,"containerName":"Scalar","name":"Set","line":269},{"containerName":"Scalar","kind":12,"name":"Set","line":271},{"kind":12,"range":{"start":{"line":302,"character":0},"end":{"character":9999,"line":304}},"line":302,"signature":{"parameters":[{"label":"$tree"},{"label":"$labels"}],"documentation":"1;\n# $Id: Compatible.pm 16123 2009-09-17 12:57:27Z cjfields $\n#\n# BioPerl module for Bio::Tree::Compatible\n#\n# Please direct questions and support issues to <bioperl-l@bioperl.org> \n#\n# Cared for by Gabriel Valiente <valiente@lsi.upc.edu>\n#\n# Copyright Gabriel Valiente\n#\n# You may distribute this module under the same terms as Perl itself\n\n# POD documentation - main docs before the code\n\n=head1 NAME\n\nBio::Tree::Compatible - Testing compatibility of phylogenetic trees\nwith nested taxa.\n\n=head1 SYNOPSIS\n\n  use Bio::Tree::Compatible;\n  use Bio::TreeIO;\n  my $input = Bio::TreeIO->new('-format' => 'newick',\n                               '-file'   => 'input.tre');\n  my $t1 = $input->next_tree;\n  my $t2 = $input->next_tree;\n\n  my ($incompat, $ilabels, $inodes) = Bio::Tree::Compatible::is_compatible($t1,$t2);\n  if ($incompat) {\n    my %cluster1 = %{ Bio::Tree::Compatible::cluster_representation($t1) };\n    my %cluster2 = %{ Bio::Tree::Compatible::cluster_representation($t2) };\n    print \"incompatible trees\\n\";\n    if (scalar(@$ilabels)) {\n      foreach my $label (@$ilabels) {\n        my $node1 = $t1->find_node(-id => $label);\n        my $node2 = $t2->find_node(-id => $label);\n        my @c1 = sort @{ $cluster1{$node1} };\n        my @c2 = sort @{ $cluster2{$node2} };\n        print \"label $label\";\n        print \" cluster\"; map { print \" \",$_ } @c1;\n        print \" cluster\"; map { print \" \",$_ } @c2; print \"\\n\";\n      }\n    }\n    if (scalar(@$inodes)) {\n      while (@$inodes) {\n        my $node1 = shift @$inodes;\n        my $node2 = shift @$inodes;\n        my @c1 = sort @{ $cluster1{$node1} };\n        my @c2 = sort @{ $cluster2{$node2} };\n        print \"cluster\"; map { print \" \",$_ } @c1;\n        print \" properly intersects cluster\";\n        map { print \" \",$_ } @c2; print \"\\n\";\n      }\n    }\n  } else {\n    print \"compatible trees\\n\";\n  }\n\n=head1 DESCRIPTION\n\nNB: This module has exclusively class methods that work on Bio::Tree::TreeI\nobjects. An instance of Bio::Tree::Compatible cannot itself represent a tree,\nand so typically there is no need to create one.\n\nBio::Tree::Compatible is a Perl tool for testing compatibility of\nphylogenetic trees with nested taxa represented as Bio::Tree::Tree\nobjects. It is based on a recent characterization of ancestral\ncompatibility of semi-labeled trees in terms of their cluster\nrepresentations.\n\nA semi-labeled tree is a phylogenetic tree with some of its internal\nnodes labeled, and it can represent a classification tree as well as a\nphylogenetic tree with nested taxa, with labeled internal nodes\ncorresponding to taxa at a higher level of aggregation or nesting than\nthat of their descendents.\n\nTwo semi-labeled trees are compatible if their topological\nrestrictions to the common labels are such that for each node label,\nthe smallest clusters containing it in each of the trees coincide and,\nfurthermore, no cluster in one of the trees properly intersects a\ncluster of the other tree.\n\nFuture extensions of Bio::Tree::Compatible include a\nBio::Tree::Supertree module for combining compatible phylogenetic\ntrees with nested taxa into a common supertree.\n\n=head1 FEEDBACK\n\n=head2 Mailing Lists\n\nUser feedback is an integral part of the evolution of this and other\nBioperl modules. Send your comments and suggestions preferably to the\nBioperl mailing list.  Your participation is much appreciated.\n\n  bioperl-l@bioperl.org                  - General discussion\n  http://bioperl.org/wiki/Mailing_lists  - About the mailing lists\n\n=head2 Support \n\nPlease direct usage questions or support issues to the mailing list:\n\nI<bioperl-l@bioperl.org>\n\nrather than to the module maintainer directly. Many experienced and \nreponsive experts will be able look at the problem and quickly \naddress it. Please include a thorough description of the problem \nwith code and data examples if at all possible.\n\n=head2 Reporting Bugs\n\nReport bugs to the Bioperl bug tracking system to help us keep track\nof the bugs and their resolution. Bug reports can be submitted via the\nweb:\n\n  http://bugzilla.open-bio.org/\n\n=head1 SEE ALSO\n\n\n* * Philip Daniel and Charles Semple. Supertree Algorithms for\nNested Taxa. In: Olaf R. P. Bininda-Emonds (ed.) Phylogenetic\nSupertrees: Combining Information to Reveal the Tree of Life,\nI<Computational Biology>, vol. 4, chap. 7, pp. 151-171. Kluwer (2004).\n\n* * Charles Semple, Philip Daniel, Wim Hordijk, Roderic\nD. M. Page, and Mike Steel: Supertree Algorithms for Ancestral\nDivergence Dates and Nested Taxa. Bioinformatics B<20>(15), 2355-2360\n(2004).\n\n* * Merce Llabres, Jairo Rocha, Francesc Rossello, and Gabriel\nValiente: On the Ancestral Compatibility of Two Phylogenetic Trees\nwith Nested Taxa. J. Math. Biol. B<53>(3), 340-364 (2006).\n\n\n=head1 AUTHOR - Gabriel Valiente\n\nEmail valiente@lsi.upc.edu\n\n=head1 APPENDIX\n\nThe rest of the documentation details each of the object methods.\n\n\npackage Bio::Tree::Compatible;\nuse strict;\n\n# Object preamble - inherits from Bio::Root::Root\n\nuse Set::Scalar;\n\nuse base qw(Bio::Root::Root);\n\n=head2 postorder_traversal\n\n Title   : postorder_traversal\n Usage   : my @nodes = @{ $tree->postorder_traversal }\n Function: Return list of nodes in postorder\n Returns : reference to array of Bio::Tree::Node\n Args    : none\n\nFor example, the postorder traversal of the tree\nC<(((A,B)C,D),(E,F,G));> is a reference to an array of nodes with\ninternal_id 0 through 9, because the Newick standard representation\nfor phylogenetic trees is based on a postorder traversal.\n\n          +---A                    +---0\n          |                        |\n  +---+---C                +---4---2\n  |   |   |                |   |   |\n  |   |   +---B            |   |   +---1\n  |   |                    |   |\n  +   +-------D            9   +-------3\n  |                        |\n  |     +-----E            |     +-----5\n  |     |                  |     |\n  +-----+-----F            +-----8-----6\n        |                        |\n        +-----G                  +-----7\n\n\nsub postorder_traversal {\n  my($self) = @_;\n  my @stack;\n  my @queue;\n  push @stack, $self->get_root_node;\n  while (@stack) {\n    my $node = pop @stack;\n    push @queue, $node;\n    foreach my $child ($node->each_Descendent(-sortby => 'internal_id')) {\n      push @stack, $child;\n    }\n  }\n  my @postorder = reverse @queue;\n  return \\@postorder;\n}\n\n=head2 cluster_representation\n\n Title   : cluster_representation\n Usage   : my %cluster = %{ $tree->cluster_representation }\n Function: Compute the cluster representation of a tree\n Returns : reference to hash of array of string indexed by\n           Bio::Tree::Node\n Args    : none\n\nFor example, the cluster representation of the tree\nC<(((A,B)C,D),(E,F,G));> is a reference to a hash associating an array\nof string (descendent labels) to each node, as follows:\n\n  0 --> [A]\n  1 --> [B]\n  2 --> [A,B,C]\n  3 --> [D]\n  4 --> [A,B,C,D]\n  5 --> [E]\n  6 --> [F]\n  7 --> [G]\n  8 --> [E,F,G]\n  9 --> [A,B,C,D,E,F,G]\n\n\nsub cluster_representation {\n  my ($tree) = @_;\n  my %cluster;\n  my @postorder = @{ postorder_traversal($tree) };\n  foreach my $node ( @postorder ) {\n    my @labeled = map { $_->id } grep { $_->id } $node->get_Descendents;\n    push @labeled, $node->id if $node->id;\n    $cluster{$node} = \\@labeled;\n  }\n  return \\%cluster;\n}\n\n=head2 common_labels\n\n Title   : common_labels\n Usage   : my $labels = $tree1->common_labels($tree2);\n Function: Return set of common node labels\n Returns : Set::Scalar\n Args    : Bio::Tree::Tree\n\nFor example, the common labels of the tree C<(((A,B)C,D),(E,F,G));>\nand the tree C<((A,B)H,E,(J,(K)G)I);> are: C<[A,B,E,G]>.\n\n          +---A                 +---A\n          |                     |\n  +---+---C             +-------H\n  |   |   |             |       |\n  |   |   +---B         |       +---B\n  |   |                 |\n  +   +-------D         +-----------E\n  |                     |\n  |     +-----E         |   +-------J\n  |     |               |   |\n  +-----+-----F         +---I\n        |                   |\n        +-----G             +---G---K\n\n\nsub common_labels {\n  my($self,$arg) = @_;\n  my @labels1 = map { $_->id } grep { $_->id } $self->get_nodes;\n  my $common = Set::Scalar->new( @labels1 );\n  my @labels2 = map { $_->id } grep { $_->id } $arg->get_nodes;\n  my $temp = Set::Scalar->new( @labels2 );\n  return $common->intersection($temp);\n}\n\n=head2 topological_restriction\n\n Title   : topological_restriction\n Usage   : $tree->topological_restriction($labels)\n Function: Compute the topological restriction of a tree to a subset\n           of node labels\n Returns : Bio::Tree::Tree\n Args    : Set::Scalar\n\nFor example, the topological restrictions of each of the trees\nC<(((A,B)C,D),(E,F,G));> and C<((A,B)H,E,(J,(K)G)I);> to the labels\nC<[A,B,E,G]> are as follows:\n\n          +---A             +---A\n          |                 |\n  +---+---+             +---+\n  |       |             |   |\n  |       +---B         |   +---B\n  +                     |\n  |       +---E         +-------E\n  |       |             |\n  +-------+             +---+---G\n          |\n          +---G","label":"topological_restriction($tree,$labels)"},"name":"topological_restriction","containerName":"main::","children":[{"definition":"my","line":303,"localvar":"my","kind":13,"containerName":"topological_restriction","name":"$tree"},{"name":"$labels","kind":13,"containerName":"topological_restriction","line":303},{"definition":"my","localvar":"my","kind":13,"containerName":"topological_restriction","name":"$node","line":304},{"line":304,"containerName":"topological_restriction","kind":13,"name":"$tree"}],"detail":"($tree,$labels)","definition":"sub"},{"line":305,"name":"%node","kind":13,"containerName":null},{"name":"@cluster","containerName":null,"localvar":"my","kind":13,"line":306,"definition":"my"},{"containerName":"main::","kind":12,"name":"id","line":306},{"name":"id","containerName":"main::","kind":12,"line":306},{"kind":13,"containerName":null,"name":"$node","line":306},{"line":306,"name":"get_Descendents","containerName":"main::","kind":12},{"line":307,"name":"@cluster","containerName":null,"kind":13},{"line":307,"kind":13,"containerName":null,"name":"$node"},{"line":307,"name":"id","kind":12,"containerName":"main::"},{"line":307,"kind":13,"containerName":null,"name":"$node"},{"line":307,"name":"id","containerName":"main::","kind":12},{"line":308,"localvar":"my","kind":13,"containerName":null,"name":"$cluster","definition":"my"},{"line":308,"name":"Set","containerName":"Scalar","kind":12},{"containerName":"main::","kind":12,"name":"new","line":308},{"name":"@cluster","containerName":null,"kind":13,"line":308},{"line":309,"name":"$cluster","containerName":null,"kind":13},{"line":309,"name":"is_disjoint","kind":12,"containerName":"main::"},{"line":309,"containerName":null,"kind":13,"name":"%labels"},{"line":310,"kind":13,"containerName":null,"name":"$tree"},{"name":"remove_Node","kind":12,"containerName":"main::","line":310},{"containerName":null,"kind":13,"name":"%node","line":310},{"line":312,"kind":13,"containerName":null,"name":"$node"},{"line":312,"containerName":"main::","kind":12,"name":"id"},{"line":312,"kind":13,"containerName":null,"name":"$labels"},{"line":312,"kind":12,"containerName":"main::","name":"has"},{"line":312,"name":"$node","containerName":null,"kind":13},{"line":312,"containerName":"main::","kind":12,"name":"id"},{"kind":13,"containerName":null,"name":"%node","line":313},{"kind":12,"range":{"start":{"line":385,"character":0},"end":{"character":9999,"line":390}},"line":385,"signature":{"label":"is_compatible($tree1,$tree2)","parameters":[{"label":"$tree1"},{"label":"$tree2"}],"documentation":"1;\n# $Id: Compatible.pm 16123 2009-09-17 12:57:27Z cjfields $\n#\n# BioPerl module for Bio::Tree::Compatible\n#\n# Please direct questions and support issues to <bioperl-l@bioperl.org> \n#\n# Cared for by Gabriel Valiente <valiente@lsi.upc.edu>\n#\n# Copyright Gabriel Valiente\n#\n# You may distribute this module under the same terms as Perl itself\n\n# POD documentation - main docs before the code\n\n=head1 NAME\n\nBio::Tree::Compatible - Testing compatibility of phylogenetic trees\nwith nested taxa.\n\n=head1 SYNOPSIS\n\n  use Bio::Tree::Compatible;\n  use Bio::TreeIO;\n  my $input = Bio::TreeIO->new('-format' => 'newick',\n                               '-file'   => 'input.tre');\n  my $t1 = $input->next_tree;\n  my $t2 = $input->next_tree;\n\n  my ($incompat, $ilabels, $inodes) = Bio::Tree::Compatible::is_compatible($t1,$t2);\n  if ($incompat) {\n    my %cluster1 = %{ Bio::Tree::Compatible::cluster_representation($t1) };\n    my %cluster2 = %{ Bio::Tree::Compatible::cluster_representation($t2) };\n    print \"incompatible trees\\n\";\n    if (scalar(@$ilabels)) {\n      foreach my $label (@$ilabels) {\n        my $node1 = $t1->find_node(-id => $label);\n        my $node2 = $t2->find_node(-id => $label);\n        my @c1 = sort @{ $cluster1{$node1} };\n        my @c2 = sort @{ $cluster2{$node2} };\n        print \"label $label\";\n        print \" cluster\"; map { print \" \",$_ } @c1;\n        print \" cluster\"; map { print \" \",$_ } @c2; print \"\\n\";\n      }\n    }\n    if (scalar(@$inodes)) {\n      while (@$inodes) {\n        my $node1 = shift @$inodes;\n        my $node2 = shift @$inodes;\n        my @c1 = sort @{ $cluster1{$node1} };\n        my @c2 = sort @{ $cluster2{$node2} };\n        print \"cluster\"; map { print \" \",$_ } @c1;\n        print \" properly intersects cluster\";\n        map { print \" \",$_ } @c2; print \"\\n\";\n      }\n    }\n  } else {\n    print \"compatible trees\\n\";\n  }\n\n=head1 DESCRIPTION\n\nNB: This module has exclusively class methods that work on Bio::Tree::TreeI\nobjects. An instance of Bio::Tree::Compatible cannot itself represent a tree,\nand so typically there is no need to create one.\n\nBio::Tree::Compatible is a Perl tool for testing compatibility of\nphylogenetic trees with nested taxa represented as Bio::Tree::Tree\nobjects. It is based on a recent characterization of ancestral\ncompatibility of semi-labeled trees in terms of their cluster\nrepresentations.\n\nA semi-labeled tree is a phylogenetic tree with some of its internal\nnodes labeled, and it can represent a classification tree as well as a\nphylogenetic tree with nested taxa, with labeled internal nodes\ncorresponding to taxa at a higher level of aggregation or nesting than\nthat of their descendents.\n\nTwo semi-labeled trees are compatible if their topological\nrestrictions to the common labels are such that for each node label,\nthe smallest clusters containing it in each of the trees coincide and,\nfurthermore, no cluster in one of the trees properly intersects a\ncluster of the other tree.\n\nFuture extensions of Bio::Tree::Compatible include a\nBio::Tree::Supertree module for combining compatible phylogenetic\ntrees with nested taxa into a common supertree.\n\n=head1 FEEDBACK\n\n=head2 Mailing Lists\n\nUser feedback is an integral part of the evolution of this and other\nBioperl modules. Send your comments and suggestions preferably to the\nBioperl mailing list.  Your participation is much appreciated.\n\n  bioperl-l@bioperl.org                  - General discussion\n  http://bioperl.org/wiki/Mailing_lists  - About the mailing lists\n\n=head2 Support \n\nPlease direct usage questions or support issues to the mailing list:\n\nI<bioperl-l@bioperl.org>\n\nrather than to the module maintainer directly. Many experienced and \nreponsive experts will be able look at the problem and quickly \naddress it. Please include a thorough description of the problem \nwith code and data examples if at all possible.\n\n=head2 Reporting Bugs\n\nReport bugs to the Bioperl bug tracking system to help us keep track\nof the bugs and their resolution. Bug reports can be submitted via the\nweb:\n\n  http://bugzilla.open-bio.org/\n\n=head1 SEE ALSO\n\n\n* * Philip Daniel and Charles Semple. Supertree Algorithms for\nNested Taxa. In: Olaf R. P. Bininda-Emonds (ed.) Phylogenetic\nSupertrees: Combining Information to Reveal the Tree of Life,\nI<Computational Biology>, vol. 4, chap. 7, pp. 151-171. Kluwer (2004).\n\n* * Charles Semple, Philip Daniel, Wim Hordijk, Roderic\nD. M. Page, and Mike Steel: Supertree Algorithms for Ancestral\nDivergence Dates and Nested Taxa. Bioinformatics B<20>(15), 2355-2360\n(2004).\n\n* * Merce Llabres, Jairo Rocha, Francesc Rossello, and Gabriel\nValiente: On the Ancestral Compatibility of Two Phylogenetic Trees\nwith Nested Taxa. J. Math. Biol. B<53>(3), 340-364 (2006).\n\n\n=head1 AUTHOR - Gabriel Valiente\n\nEmail valiente@lsi.upc.edu\n\n=head1 APPENDIX\n\nThe rest of the documentation details each of the object methods.\n\n\npackage Bio::Tree::Compatible;\nuse strict;\n\n# Object preamble - inherits from Bio::Root::Root\n\nuse Set::Scalar;\n\nuse base qw(Bio::Root::Root);\n\n=head2 postorder_traversal\n\n Title   : postorder_traversal\n Usage   : my @nodes = @{ $tree->postorder_traversal }\n Function: Return list of nodes in postorder\n Returns : reference to array of Bio::Tree::Node\n Args    : none\n\nFor example, the postorder traversal of the tree\nC<(((A,B)C,D),(E,F,G));> is a reference to an array of nodes with\ninternal_id 0 through 9, because the Newick standard representation\nfor phylogenetic trees is based on a postorder traversal.\n\n          +---A                    +---0\n          |                        |\n  +---+---C                +---4---2\n  |   |   |                |   |   |\n  |   |   +---B            |   |   +---1\n  |   |                    |   |\n  +   +-------D            9   +-------3\n  |                        |\n  |     +-----E            |     +-----5\n  |     |                  |     |\n  +-----+-----F            +-----8-----6\n        |                        |\n        +-----G                  +-----7\n\n\nsub postorder_traversal {\n  my($self) = @_;\n  my @stack;\n  my @queue;\n  push @stack, $self->get_root_node;\n  while (@stack) {\n    my $node = pop @stack;\n    push @queue, $node;\n    foreach my $child ($node->each_Descendent(-sortby => 'internal_id')) {\n      push @stack, $child;\n    }\n  }\n  my @postorder = reverse @queue;\n  return \\@postorder;\n}\n\n=head2 cluster_representation\n\n Title   : cluster_representation\n Usage   : my %cluster = %{ $tree->cluster_representation }\n Function: Compute the cluster representation of a tree\n Returns : reference to hash of array of string indexed by\n           Bio::Tree::Node\n Args    : none\n\nFor example, the cluster representation of the tree\nC<(((A,B)C,D),(E,F,G));> is a reference to a hash associating an array\nof string (descendent labels) to each node, as follows:\n\n  0 --> [A]\n  1 --> [B]\n  2 --> [A,B,C]\n  3 --> [D]\n  4 --> [A,B,C,D]\n  5 --> [E]\n  6 --> [F]\n  7 --> [G]\n  8 --> [E,F,G]\n  9 --> [A,B,C,D,E,F,G]\n\n\nsub cluster_representation {\n  my ($tree) = @_;\n  my %cluster;\n  my @postorder = @{ postorder_traversal($tree) };\n  foreach my $node ( @postorder ) {\n    my @labeled = map { $_->id } grep { $_->id } $node->get_Descendents;\n    push @labeled, $node->id if $node->id;\n    $cluster{$node} = \\@labeled;\n  }\n  return \\%cluster;\n}\n\n=head2 common_labels\n\n Title   : common_labels\n Usage   : my $labels = $tree1->common_labels($tree2);\n Function: Return set of common node labels\n Returns : Set::Scalar\n Args    : Bio::Tree::Tree\n\nFor example, the common labels of the tree C<(((A,B)C,D),(E,F,G));>\nand the tree C<((A,B)H,E,(J,(K)G)I);> are: C<[A,B,E,G]>.\n\n          +---A                 +---A\n          |                     |\n  +---+---C             +-------H\n  |   |   |             |       |\n  |   |   +---B         |       +---B\n  |   |                 |\n  +   +-------D         +-----------E\n  |                     |\n  |     +-----E         |   +-------J\n  |     |               |   |\n  +-----+-----F         +---I\n        |                   |\n        +-----G             +---G---K\n\n\nsub common_labels {\n  my($self,$arg) = @_;\n  my @labels1 = map { $_->id } grep { $_->id } $self->get_nodes;\n  my $common = Set::Scalar->new( @labels1 );\n  my @labels2 = map { $_->id } grep { $_->id } $arg->get_nodes;\n  my $temp = Set::Scalar->new( @labels2 );\n  return $common->intersection($temp);\n}\n\n=head2 topological_restriction\n\n Title   : topological_restriction\n Usage   : $tree->topological_restriction($labels)\n Function: Compute the topological restriction of a tree to a subset\n           of node labels\n Returns : Bio::Tree::Tree\n Args    : Set::Scalar\n\nFor example, the topological restrictions of each of the trees\nC<(((A,B)C,D),(E,F,G));> and C<((A,B)H,E,(J,(K)G)I);> to the labels\nC<[A,B,E,G]> are as follows:\n\n          +---A             +---A\n          |                 |\n  +---+---+             +---+\n  |       |             |   |\n  |       +---B         |   +---B\n  +                     |\n  |       +---E         +-------E\n  |       |             |\n  +-------+             +---+---G\n          |\n          +---G\n\n\nsub topological_restriction {\n  my ($tree, $labels) = @_;\n  for my $node ( @{ postorder_traversal($tree) } ) {\n    unless (ref($node)) { # skip $node if already removed\n      my @cluster = map { $_->id } grep { $_->id } $node->get_Descendents;\n      push @cluster, $node->id if $node->id;\n      my $cluster = Set::Scalar->new(@cluster);\n      if ($cluster->is_disjoint($labels)) {\n        $tree->remove_Node($node);\n      } else {\n        if ($node->id and not $labels->has($node->id)) {\n          $node->{'_id'} = undef;\n        }\n      }\n    }\n  }\n}\n\n=head2 is_compatible\n\n Title   : is_compatible\n Usage   : $tree1->is_compatible($tree2)\n Function: Test compatibility of two trees\n Returns : boolean\n Args    : Bio::Tree::Tree\n\nFor example, the topological restrictions of the trees\nC<(((A,B)C,D),(E,F,G));> and C<((A,B)H,E,(J,(K)G)I);> to their common\nlabels, C<[A,B,E,G]>, are compatible. The respective cluster\nrepresentations are as follows:\n\n  [A]                  [A]\n  [B]                  [B]\n  [E]                  [E]\n  [G]                  [G]\n  [A,B]                [A,B]\n  [E,G]                [A,B,E,G]\n  [A,B,E,G]\n\nAs a second example, the trees C<(A,B);> and C<((B)A);> are\nincompatible. Their respective cluster representations are as follows:\n\n  [A]                  [B]\n  [B]                  [A,B]\n  [A,B]\n\nThe reason is, the smallest cluster containing label C<A> is C<[A]> in\nthe first tree but C<[A,B]> in the second tree.\n\n +---A         A---B\n |\n +\n |\n +---B\n\nAs a second example, the trees C<(((B,A),C),D);> and C<((A,(D,B)),C);>\nare also incompatible. Their respective cluster representations are as\nfollows:\n\n  [A]                  [A]\n  [B]                  [B]\n  [C]                  [C]\n  [D]                  [D]\n  [A,B]                [B,D]\n  [A,B,C]              [A,B,D]\n  [A,B,C,D]            [A,B,C,D]\n\nThe reason is, cluster C<[A,B]> properly intersects cluster\nC<[B,D]>. There are further incompatibilities between these trees:\nC<[A,B,C]> properly intersects both C<[B,D]> and C<[A,B,D]>.\n\n          +---B             +-------A\n          |                 |\n      +---+             +---+   +---D\n      |   |             |   |   |\n  +---+   +---A         |   +---+\n  |   |                 +       |\n  +   +-------C         |       +---B\n  |                     |\n  +-----------D         +-----------C"},"name":"is_compatible","containerName":"main::","children":[{"definition":"my","line":386,"localvar":"my","containerName":"is_compatible","kind":13,"name":"$tree1"},{"kind":13,"containerName":"is_compatible","name":"$tree2","line":386},{"definition":"my","name":"$common","kind":13,"localvar":"my","containerName":"is_compatible","line":387},{"line":387,"name":"$tree1","kind":13,"containerName":"is_compatible"},{"name":"$tree2","kind":13,"containerName":"is_compatible","line":387},{"line":388,"kind":13,"containerName":"is_compatible","name":"$tree1"},{"line":388,"containerName":"is_compatible","kind":13,"name":"$common"},{"line":389,"containerName":"is_compatible","kind":13,"name":"$tree2"},{"kind":13,"containerName":"is_compatible","name":"$common","line":389},{"line":390,"kind":13,"localvar":"my","containerName":"is_compatible","name":"@postorder1","definition":"my"},{"name":"$tree1","containerName":"is_compatible","kind":13,"line":390}],"detail":"($tree1,$tree2)","definition":"sub"},{"line":387,"name":"Bio","containerName":"Tree::Compatible::common_labels","kind":12},{"line":388,"name":"Bio","containerName":"Tree::Compatible::topological_restriction","kind":12},{"name":"Bio","containerName":"Tree::Compatible::topological_restriction","kind":12,"line":389},{"line":391,"name":"@postorder2","kind":13,"localvar":"my","containerName":null,"definition":"my"},{"kind":13,"containerName":null,"name":"$tree2","line":391},{"definition":"my","containerName":null,"localvar":"my","kind":13,"name":"%cluster1","line":392},{"name":"$tree1","kind":13,"containerName":null,"line":392},{"name":"%cluster2","localvar":"my","containerName":null,"kind":13,"line":393,"definition":"my"},{"line":393,"name":"$tree2","kind":13,"containerName":null},{"name":"$incompat","localvar":"my","containerName":null,"kind":13,"line":394,"definition":"my"},{"line":395,"name":"@labels","localvar":"my","kind":13,"containerName":null,"definition":"my"},{"line":396,"name":"$label","kind":13,"localvar":"my","containerName":null,"definition":"my"},{"containerName":null,"kind":13,"name":"$common","line":396},{"line":396,"name":"elements","containerName":"main::","kind":12},{"line":397,"localvar":"my","containerName":null,"kind":13,"name":"$node1","definition":"my"},{"line":397,"kind":13,"containerName":null,"name":"$tree1"},{"line":397,"kind":12,"containerName":"main::","name":"find_node"},{"line":397,"kind":13,"containerName":null,"name":"$label"},{"line":398,"containerName":null,"localvar":"my","kind":13,"name":"@labels1","definition":"my"},{"line":398,"kind":13,"containerName":null,"name":"%cluster1"},{"name":"$node1","kind":13,"containerName":null,"line":398},{"line":399,"localvar":"my","kind":13,"containerName":null,"name":"$cluster1","definition":"my"},{"line":399,"name":"Set","containerName":"Scalar","kind":12},{"name":"new","containerName":"main::","kind":12,"line":399},{"line":399,"name":"@labels1","containerName":null,"kind":13},{"line":400,"name":"$node2","containerName":null,"localvar":"my","kind":13,"definition":"my"},{"name":"$tree2","kind":13,"containerName":null,"line":400},{"line":400,"containerName":"main::","kind":12,"name":"find_node"},{"line":400,"kind":13,"containerName":null,"name":"$label"},{"definition":"my","name":"@labels2","localvar":"my","kind":13,"containerName":null,"line":401},{"kind":13,"containerName":null,"name":"%cluster2","line":401},{"containerName":null,"kind":13,"name":"$node2","line":401},{"definition":"my","kind":13,"localvar":"my","containerName":null,"name":"$cluster2","line":402},{"name":"Set","containerName":"Scalar","kind":12,"line":402},{"line":402,"name":"new","kind":12,"containerName":"main::"},{"line":402,"containerName":null,"kind":13,"name":"@labels2"},{"line":403,"name":"$cluster1","kind":13,"containerName":null},{"line":403,"name":"is_equal","containerName":"main::","kind":12},{"kind":13,"containerName":null,"name":"%cluster2","line":403},{"containerName":null,"kind":13,"name":"$incompat","line":404},{"containerName":null,"kind":13,"name":"@labels","line":405},{"line":405,"name":"$label","kind":13,"containerName":null},{"definition":"my","line":408,"name":"@nodes","kind":13,"localvar":"my","containerName":null},{"name":"$node1","containerName":null,"localvar":"my","kind":13,"line":409,"definition":"my"},{"kind":13,"containerName":null,"name":"@postorder1","line":409},{"definition":"my","name":"@labels1","localvar":"my","containerName":null,"kind":13,"line":410},{"name":"%cluster1","containerName":null,"kind":13,"line":410},{"containerName":null,"kind":13,"name":"$node1","line":410},{"line":411,"name":"$cluster1","kind":13,"localvar":"my","containerName":null,"definition":"my"},{"name":"Set","kind":12,"containerName":"Scalar","line":411},{"line":411,"containerName":"main::","kind":12,"name":"new"},{"kind":13,"containerName":null,"name":"@labels1","line":411},{"line":412,"name":"$node2","kind":13,"localvar":"my","containerName":null,"definition":"my"},{"containerName":null,"kind":13,"name":"@postorder2","line":412},{"line":413,"kind":13,"localvar":"my","containerName":null,"name":"@labels2","definition":"my"},{"containerName":null,"kind":13,"name":"%cluster2","line":413},{"line":413,"containerName":null,"kind":13,"name":"$node2"},{"definition":"my","line":414,"localvar":"my","kind":13,"containerName":null,"name":"$cluster2"},{"containerName":"Scalar","kind":12,"name":"Set","line":414},{"line":414,"name":"new","kind":12,"containerName":"main::"},{"containerName":null,"kind":13,"name":"@labels2","line":414},{"line":415,"name":"$cluster1","kind":13,"containerName":null},{"line":415,"name":"is_properly_intersecting","kind":12,"containerName":"main::"},{"name":"%cluster2","containerName":null,"kind":13,"line":415},{"containerName":null,"kind":13,"name":"$incompat","line":416},{"name":"@nodes","containerName":null,"kind":13,"line":417},{"name":"$node1","containerName":null,"kind":13,"line":417},{"line":417,"name":"$node2","kind":13,"containerName":null},{"kind":13,"containerName":null,"name":"$incompat","line":421},{"line":421,"kind":13,"containerName":null,"name":"@labels"},{"line":421,"containerName":null,"kind":13,"name":"@nodes"}]}